<?php
/*
Plugin Name: Digital Onion Custom Portal
Description: 投稿をトップページやアーカイブ一覧にカード型で表示できるプラグインです。接頭語を除去したクリーンなアーカイブタイトル表示に対応しています。
Version: 10.0
Author: Awaji_Tamao
*/

if (!defined('ABSPATH')) exit;

/**
 * 1. 管理画面に設定メニューを追加
 */
add_action('admin_menu', function() {
    add_menu_page(
        'Portal設定', 'Portal設定', 'manage_options', 'do-portal-settings', 'do_portal_settings_page', 'dashicons-layout', 25
    );
});

add_action('admin_init', function() {
    register_setting('do-portal-group', 'do_top_cols');
    register_setting('do-portal-group', 'do_top_rows');
    register_setting('do-portal-group', 'do_archive_cols');
    register_setting('do-portal-group', 'do_archive_rows');
    register_setting('do-portal-group', 'do_archive_page_id');
    register_setting('do-portal-group', 'do_show_sidebar');
});

/**
 * 2. 設定画面のHTML
 */
function do_portal_settings_page() {
    ?>
    <div class="wrap">
        <h1>Digital Onion Custom Portal 設定</h1>
        <form method="post" action="options.php">
            <?php settings_fields('do-portal-group'); ?>
            <table class="form-table">
                <tr><th colspan="2"><h3>【トップページ設定】 [custom_list_button]</h3></th></tr>
                <tr>
                    <th>カラム数</th>
                    <td>
                        <select name="do_top_cols">
                            <option value="33.333" <?php selected(get_option('do_top_cols'), '33.333'); ?>>3カラム</option>
                            <option value="25" <?php selected(get_option('do_top_cols'), '25'); ?>>4カラム</option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th>表示行数</th>
                    <td><input type="number" name="do_top_rows" value="<?php echo esc_attr(get_option('do_top_rows', 1)); ?>" min="1" max="5"> 行</td>
                </tr>
                <tr><th colspan="2"><h3>【一覧ページ設定】 [custom_archive] ＆ 自動適用ページ</h3></th></tr>
                <tr>
                    <th>カラム数</th>
                    <td>
                        <select name="do_archive_cols">
                            <option value="33.333" <?php selected(get_option('do_archive_cols'), '33.333'); ?>>3カラム</option>
                            <option value="25" <?php selected(get_option('do_archive_cols'), '25'); ?>>4カラム</option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th>1ページあたりの行数</th>
                    <td><input type="number" name="do_archive_rows" value="<?php echo esc_attr(get_option('do_archive_rows', 4)); ?>" min="1" max="10"> 行</td>
                </tr>
                <tr>
                    <th>一覧用固定ページ</th>
                    <td>
                        <?php wp_dropdown_pages(array('name' => 'do_archive_page_id', 'selected' => get_option('do_archive_page_id'), 'show_option_none' => '未選択', 'option_none_value' => '0')); ?>
                    </td>
                </tr>
                <tr>
                    <th>サイドバー表示</th>
                    <td>
                        <label>
                            <input type="checkbox" name="do_show_sidebar" value="1" <?php checked(get_option('do_show_sidebar'), '1'); ?>>
                            自動適用ページ（アーカイブ等）でサイドバーを表示する
                        </label>
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

/**
 * 3. スタイル出力
 */
add_action('wp_head', function() {
    $t_col = get_option('do_top_cols', '25');
    $a_col = get_option('do_archive_cols', '25');
    $has_sidebar = get_option('do_show_sidebar') == '1';
    ?>
    <style>
        .do-grid { display: flex; flex-wrap: wrap; gap: 20px; margin-bottom: 30px; }
        .do-card { background: #fff; border: 1px solid #eee; border-radius: 8px; overflow: hidden; display: flex; flex-direction: column; text-decoration: none !important; color: #333 !important; transition: 0.3s; box-shadow: 0 2px 5px rgba(0,0,0,0.05); position: relative; }
        
        .do-portal-top .do-card { width: calc(<?php echo $t_col; ?>% - <?php echo ($t_col == '25') ? '15px' : '14px'; ?>); }
        .do-portal-archive .do-card { width: calc(<?php echo $a_col; ?>% - <?php echo ($a_col == '25') ? '15px' : '14px'; ?>); }
        .is-custom-archive-page .do-card { width: calc(<?php echo $a_col; ?>% - <?php echo ($a_col == '25') ? '15px' : '14px'; ?>); }

        .do-card:hover, .do-btn:hover { transform: translateY(-3px); box-shadow: 0 5px 15px rgba(0,0,0,0.1); }
        .do-thumb-wrap { position: relative; width: 100%; aspect-ratio: 16/9; overflow: hidden; }
        .do-thumb { width: 100%; height: 100%; object-fit: cover; }
        .do-cat-label { position: absolute; top: 10px; left: 10px; background: rgba(51, 51, 51, 0.8); color: #fff; padding: 2px 8px; font-size: 0.7em; border-radius: 4px; z-index: 10; font-weight: bold; }
        .do-body { padding: 15px; flex-grow: 1; }
        .do-title { font-size: 1.0em; font-weight: bold; margin-bottom: 5px; line-height: 1.4; display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow: hidden; }
        .do-date { font-size: 0.75em; color: #999; margin-bottom: 10px; display: block; }
        .do-excerpt { font-size: 0.8em; color: #666; line-height: 1.5; }
        .do-btn-wrap { text-align: center; margin: 30px 0; }
        .do-btn { display: inline-block; background: #333; color: #fff !important; padding: 12px 40px; border-radius: 30px; font-weight: bold; text-decoration: none !important; transition: 0.3s; box-shadow: 0 2px 5px rgba(0,0,0,0.05); }
        .do-nav-wrapper { width: 100% !important; text-align: center !important; margin: 40px 0 !important; }
        .do-page-item { display: inline-block !important; min-width: 40px !important; line-height: 40px !important; margin: 0 5px !important; background: #fff !important; border: 1px solid #ddd !important; color: #333 !important; text-decoration: none !important; border-radius: 4px; text-align: center !important; }
        .do-page-item.is-current { background: #333 !important; color: #fff !important; }

        /* サイドバー有効時のレイアウト */
        <?php if($has_sidebar): ?>
        .do-main-wrapper { display: flex; gap: 30px; max-width: 1200px; margin: 40px auto; padding: 0 20px; }
        .do-content-area { flex-grow: 1; }
        .do-sidebar-area { width: 300px; flex-shrink: 0; }
        @media screen and (max-width: 900px) { .do-main-wrapper { flex-direction: column; } .do-sidebar-area { width: 100%; } }
        <?php else: ?>
        .do-main-wrapper { max-width: 1200px; margin: 40px auto; padding: 0 20px; }
        <?php endif; ?>

        @media screen and (max-width: 1024px) { .do-card { width: calc(50% - 10px) !important; } }
        @media screen and (max-width: 600px) { .do-card { width: 100% !important; } }
    </style>
    <?php
});

/**
 * 4. 投稿取得ロジック
 */
function do_render_logic($cols_percent, $rows, $is_archive = false, $query_args = array()) {
    $cols_num = ($cols_percent == '25') ? 4 : 3;
    $count = $cols_num * $rows;
    $paged = max(1, get_query_var('paged'), get_query_var('page'));
    
    $default_args = array('post_type' => 'post', 'posts_per_page' => $count, 'paged' => $paged, 'post_status' => 'publish');
    $args = array_merge($default_args, $query_args);
    
    $query = new WP_Query($args);
    $html = '<div class="do-grid">';
    
    if ($query->have_posts()) {
        while ($query->have_posts()) {
            $query->the_post();
            $thumb = get_the_post_thumbnail_url(get_the_ID(), 'medium') ?: 'https://via.placeholder.com/400x225?text=No+Image';
            $categories = get_the_category();
            $cat_html = (!empty($categories)) ? '<span class="do-cat-label">'.esc_html($categories[0]->name).'</span>' : '';

            $html .= sprintf(
                '<a href="%s" class="do-card"><div class="do-thumb-wrap">%s<img src="%s" class="do-thumb"></div><div class="do-body"><div class="do-title">%s</div><span class="do-date">%s</span><div class="do-excerpt">%s</div></div></a>',
                get_permalink(), $cat_html, esc_url($thumb), get_the_title(), get_the_date(), wp_trim_words(get_the_excerpt(), 80, '...')
            );
        }
        wp_reset_postdata();
    } else {
        $html .= '<p>投稿が見つかりませんでした。</p>';
    }
    $html .= '</div>';
    return array('html' => $html, 'max' => $query->max_num_pages, 'paged' => $paged);
}

/**
 * 5. ショートコードとページネーション
 */
add_shortcode('custom_list_button', function() {
    $res = do_render_logic(get_option('do_top_cols', '25'), get_option('do_top_rows', 1));
    $page_id = get_option('do_archive_page_id');
    $link_url = ($page_id) ? get_permalink($page_id) : home_url('/');
    $btn = '<div class="do-btn-wrap"><a href="'.esc_url($link_url).'" class="do-btn">もっと投稿を見る</a></div>';
    return '<div class="do-portal do-portal-top">'.$res['html'].$btn.'</div>';
});

add_shortcode('custom_archive', function() {
    $res = do_render_logic(get_option('do_archive_cols', '25'), get_option('do_archive_rows', 4), true);
    return render_pagination_html($res);
});

function render_pagination_html($res) {
    $pages = paginate_links(array('total' => $res['max'], 'current' => $res['paged'], 'type' => 'array', 'prev_text' => '前へ', 'next_text' => '次へ'));
    $nav = '';
    if (is_array($pages)) {
        $nav .= '<div class="do-nav-wrapper">';
        foreach ($pages as $p) {
            $class = (strpos($p, 'current') !== false) ? 'do-page-item is-current' : 'do-page-item';
            $nav .= str_replace('page-numbers', $class, $p);
        }
        $nav .= '</div>';
    }
    return '<div class="do-portal do-portal-archive">'.$res['html'].$nav.'</div>';
}

/**
 * 6. アーカイブタイトルの接頭語（カテゴリー：等）を削除
 */
add_filter('get_the_archive_title', function ($title) {
    if (is_category()) {
        $title = single_cat_title('', false);
    } elseif (is_tag()) {
        $title = single_tag_title('', false);
    } elseif (is_year()) {
        $title = get_the_date('Y年');
    } elseif (is_month()) {
        $title = get_the_date('Y年n月');
    } elseif (is_day()) {
        $title = get_the_date('Y年n月j日');
    } elseif (is_author()) {
        $title = get_the_author();
    } elseif (is_post_type_archive()) {
        $title = post_type_archive_title('', false);
    } elseif (is_tax()) {
        $title = single_term_title('', false);
    }
    return $title;
});

/**
 * 7. 自動適用（アーカイブページ強制差し替え）
 */
add_action('template_redirect', function() {
    if (is_archive()) {
        add_filter('template_include', function() {
            get_header();
            $has_sidebar = get_option('do_show_sidebar') == '1';

            echo '<div class="do-main-wrapper is-custom-archive-page">';
            echo '<div class="do-content-area">';
            
            // 接頭語のないタイトルを出力
            echo '<h1 class="archive-title" style="margin-bottom:30px;">' . get_the_archive_title() . '</h1>';
            
            $args = array();
            if (is_category()) $args['cat'] = get_queried_object_id();
            if (is_tag()) $args['tag_id'] = get_queried_object_id();
            if (is_date()) {
                if (is_year()) $args['year'] = get_query_var('year');
                if (is_month()) { $args['year'] = get_query_var('year'); $args['monthnum'] = get_query_var('monthnum'); }
            }

            $res = do_render_logic(get_option('do_archive_cols', '25'), get_option('do_archive_rows', 4), true, $args);
            echo render_pagination_html($res);
            echo '</div>';

            if ($has_sidebar) {
                echo '<div class="do-sidebar-area">';
                get_sidebar();
                echo '</div>';
            }
            echo '</div>';

            get_footer();
            return __DIR__ . '/empty.php'; 
        });
    }
});

add_action('init', function() {
    $file = __DIR__ . '/empty.php';
    if (!file_exists($file)) file_put_contents($file, '<?php // Silence is golden');
});